/*
 * Crestron Electronics
 * 
 * KeyButton uiHal interface
 *
 * Raymond Kuschan
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/input/matrix_keypad.h>
#include <linux/delay.h>
#include <linux/signal.h>
#include <asm/siginfo.h>        //siginfo

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#include "../ft5x06_ts_i2c.h"


#define HARDKEY_FILENAME           "keyBtns"
#define PROCFS_MAX_SIZE     1024

/* Crestron GPIO Matrix Keys  */
static const uint32_t vButton_matrix_keys[] = {
        // (row, column, keyEvent)
        KEY(0, 0, 0x89),
        KEY(0, 1, 0x8A),
        KEY(0, 2, 0x8F),
        KEY(0, 3, 0x86),
        KEY(0, 4, 0x83),

#if 0 //for virtual panels we only support 5 buttons
        KEY(1, 0, 0x88),
        KEY(1, 1, 0x8C),
        KEY(1, 2, 0x8E),
        KEY(1, 3, 0x85),
        KEY(1, 4, 0x82),

        KEY(2, 0, 0x87),
        KEY(2, 1, 0x8B),
        KEY(2, 2, 0x8D),
        KEY(2, 3, 0x84),
        KEY(2, 4, 0x81),
#endif
};

struct matrix_keymap_data vButton_keymap_data = {
        .keymap = vButton_matrix_keys,
        .keymap_size = ARRAY_SIZE(vButton_matrix_keys),
};

int g_virt_dbg_keys=0;


struct vButton_keypad {
        /* matrix key code map */
        unsigned short keycodes[MAX_KEY_NUM];
        struct input_dev *input_dev;
};

unsigned int gVbKeyStates = 0;
extern struct mutex *keyStateMutex; //mutex lock for gVbKeyStates

static struct vButton_keypad *gKeypad = NULL;

/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *HardKey_File;

/** * The buffer (2k) for this module */
char gHardKey_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
unsigned long gHardKey_buffer_size = 0;


/** * This funtion is called when the /proc hardkey file is read */
static ssize_t vbHardKey_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  static int finished = 0;
  char * p = gHardKey_buffer;
  int value;

  mutex_lock(keyStateMutex);
  value = gVbKeyStates;
  mutex_unlock(keyStateMutex);

  /* needed to stop from continuously printing */
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

   p += sprintf ( p, "KEYVALUES=%08X\n" , value );
   p += sprintf ( p, "NUMBER_OF_HARDKEYS=%04d\n" , NUM_OF_HARDKEYS );

   gHardKey_buffer_size = p-gHardKey_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gHardKey_buffer, gHardKey_buffer_size) ) {
                return -EFAULT;
        }

  return gHardKey_buffer_size;
}

static ssize_t vbHardKey_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
        /* no Write support */
        printk(KERN_ERR "Write not supported on this device, only for debug\n");
      /* code below is for debug only  it will override defined button presses 
                and enable some basic Android buttons like back and menu
        */
        char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;

        gHardKey_buffer_size = len;
        if (gHardKey_buffer_size > PROCFS_MAX_SIZE ) {
                gHardKey_buffer_size = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(gHardKey_buffer, buffer, gHardKey_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "DEBUG" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_virt_dbg_keys );
        }

        return gHardKey_buffer_size;

}
/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}
/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int vbHardKey_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int vbHardKey_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_HardKey_File = {
        .read    = vbHardKey_read,
        .write   = vbHardKey_write,
        .open    = vbHardKey_open_procfs,
        .release = vbHardKey_close_procfs,
};
/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
        .permission = module_permission,        /* check for permissions */
};

int uiVbuttonIntfInit(void)
{
        int error;

/* create the /proc file for HardKey bitmask */
        HardKey_File = create_proc_entry(HARDKEY_FILENAME, 0644, NULL);
        if (HardKey_File == NULL){
                printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
                       HARDKEY_FILENAME);
                return -ENOMEM;
        }
        else
        {
          //HardKey_File->owner = THIS_MODULE;
          HardKey_File->proc_iops = &Inode_Ops_File;
          HardKey_File->proc_fops = &File_Ops_HardKey_File;
          HardKey_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
          HardKey_File->uid = 0;
          HardKey_File->gid = 0;
          HardKey_File->size = 80;
       }

        return error;
}

